from abc import ABC, abstractmethod
from typing import Any


class BaseSampler(ABC):
    """
    Abstract base class for dataset samplers in HyperSence.
    All samplers must inherit from this class and implement the sample() method.
    """

    def __init__(self, dataset: Any, sample_size: int, seed: int = 42, **kwargs):
        """
        Initialize the sampler.

        Args:
            dataset (Any): The full dataset to sample from.
            sample_size (int): Number of samples to select.
            seed (int): Random seed for reproducibility.
            **kwargs: Optional additional parameters for flexibility.
        """
        self.dataset = dataset
        self.sample_size = sample_size
        self.seed = seed
        self.kwargs = kwargs  # store optional args for subclass usage

    @abstractmethod
    def sample(self):
        """
        Perform sampling and return the selected subset of the dataset.

        Returns:
            Any: The sampled subset (format depends on dataset type).
        """
        pass

    def save(self, output_path: str):
        """
        Optional method: Save the sampled subset to disk.
        Subclasses can override this if saving is needed.

        Args:
            output_path (str): Path to save the sampled dataset.
        """
        raise NotImplementedError("This sampler does not support saving.")
